// Antonello Macro 1: split

// 1 - A user runs the macro.
// 2 - The macro asks to choose a folder with RAW images.
// 3 - The macro reads every RAW image with DCRAW plugin one by one.
// 4 - Every RAW image is being split into 4 components.
// 5 - The brightest one (among 4 components) is being saved to disk in JPG format (into the 1st output folder).

// **********************************************************

REFERENCE_VALUES_FOR_GRAY_PATCHES = newArray(230, 184, 138, 92, 46, 11);

// Allow a user to disable/enable batch mode
Dialog.create("Antonello");
Dialog.addCheckbox("Batch Mode (faster processing)", true);
Dialog.show();
ENABLE_BATCH_MODE = Dialog.getCheckbox();


// **********************************************************

closeAllImages();

// 1 - Choose a folder with the RAW images to process
while (true) {
	rawImagesDir = chooseInputDir();
	
	if (countFilesInDir(rawImagesDir) == 0) {
		Dialog.create("Warning");
		Dialog.addMessage("The chosen folder doesn't contain any files!");
		Dialog.addMessage("Please, press OK and choose another folder.");
		Dialog.show();
		continue;
	}

	theNamesString = getFileNamesWithSpacesInDir(rawImagesDir);
	
	if (lengthOf(theNamesString) > 0) {
		Dialog.create("Warning");
		Dialog.addMessage("The next images contain spaces in their file names:");
		Dialog.addMessage(theNamesString);
		Dialog.addMessage("DCRAW plugin can't open such images.");
		Dialog.addMessage("Please, press OK and choose another folder.");
		Dialog.show();
	} else {
		break;
	}
}

// 2 - Create 3 output folders
rawImagesDirTrimmed = substring(rawImagesDir, 0, lengthOf(rawImagesDir)-1);
brightestDir = rawImagesDirTrimmed + "-1-BRIGHTEST_COMPONENTS" + File.separator;
registeredDir = rawImagesDirTrimmed + "-2-REGISTERED" + File.separator;
calibratedDir = rawImagesDirTrimmed + "-3-CALIBRATED" + File.separator;

File.makeDirectory(brightestDir);
File.makeDirectory(registeredDir);
File.makeDirectory(calibratedDir);

// 3 - Split every image into 4 components and save the brightest one to disk
enableBatchMode(true);
processRAWImages(rawImagesDir, brightestDir);

showMessage("Success", "Antonello-split has finished its job");

// **********************************************************

function enableBatchMode(isEnable) {
	if (ENABLE_BATCH_MODE) {
		setBatchMode(isEnable);
	}
}

function getFileNamesWithSpacesInDir(inputDir) {
	imagesNames = getFileList(inputDir);
	theString = "";

	for (i = 0; i < imagesNames.length; i++) {
	
		imagePath = inputDir + imagesNames[i];
	
		if (File.isDirectory(imagePath)) {
			continue;
		}

		isStringContainsSpace = (indexOf(imagesNames[i], " ") != -1);
		if (isStringContainsSpace) {
			theString += "\n" + imagesNames[i];
		}
	}
	
	return theString;
}

function countFilesInDir(inputDir) {
	imagesNames = getFileList(inputDir);
	filesCount = 0;

	for (i = 0; i < imagesNames.length; i++) {
	
		imagePath = inputDir + imagesNames[i];
	
		if (File.isDirectory(imagePath)) {
			continue;
		}
	
		filesCount++;
	}
	
	return filesCount;
}

function chooseInputDir() {
	while (true) {
		rawImagesDir = getDirectory("Choose a folder with the RAW images");
		isSpaceCharacterDetected = (indexOf(rawImagesDir, " ") != -1);

		if (isSpaceCharacterDetected) {
			
			Dialog.create("Warning");
			Dialog.addMessage("The chosen folder's path contains spaces!");
			Dialog.addMessage("DCRAW plugin can't open images in this folder.");
			Dialog.addMessage("Please, press OK and choose another folder.");
			Dialog.show();
			
		} else {
			break;
		}
	}

	return rawImagesDir;
}

function processRAWImages(inputDir, outputDir) {
	rawImagesNames = getFileList(inputDir);
	rawImagesNames = Array.sort(rawImagesNames);
	splitImagesNames = newArray("G1", "R", "B", "G2");
	
	for (i = 0; i < rawImagesNames.length; i++) {
	
		fullImagePath = inputDir + rawImagesNames[i];
		if (File.isDirectory(fullImagePath)) {
			continue;
		}
	
		run("DCRaw Reader...", "open='&fullImagePath' white_balance=None do_not_automatically_brighten output_colorspace=raw document_mode read_as=8-bit interpolation=[High-speed, low-quality bilinear]");
	
		Table.create("Results");
	
		// Splitting RAW image		
		id = getImageID();
		currentImageName = getTitle();
		makeColorChannel(splitImagesNames[0], 0, -1);
		run("Measure");
		
		selectImage(id);
		makeColorChannel(splitImagesNames[1], 0, 0);
		run("Measure");
		
		selectImage(id);
		makeColorChannel(splitImagesNames[2], -1, -1);
		run("Measure");
		
		selectImage(id);
		makeColorChannel(splitImagesNames[3], -1, 0);
		run("Measure");
	
		// Find an image with max brightness	
		meanValues = Table.getColumn("Mean");
		maxValueIndex = indexOfMaxValue(meanValues);		
	
		// Save the brightest component into a separate folder		
		brightestImagePath = outputDir + currentImageName;
		selectImage(splitImagesNames[maxValueIndex]);		
		saveAs("jpg", brightestImagePath);
	
		closeAllImages();
	}
	
	close("Results");
}

function indexOfMaxValue(theArray) {
	maxIndex = 0;	
	for (i = 1; i < theArray.length; i++) {
		if (theArray[i] > theArray[maxIndex]) {
			maxIndex = i;
		}
	}

	return maxIndex;
}

function makeColorChannel(name, dx, dy) {
	slices = nSlices();
	newWidth = getWidth()/2;
	newHeight = getHeight()/2;
	run("Duplicate...", "title=&name duplicate range=1-&slices");
	run("Translate...", "x=&dx y=&dy interpolation=None stack");
	run("Size...", "width=&newWidth height=&newHeight depth=&slices interpolation=None");
}

function closeAllImages() {
	while (nImages > 0) {
		selectImage(nImages);
		close();
	}
}

