// Antonello Macro 2: calibration

// 1 - A user runs the macro.
// 2 - The macro asks to choose a folder with RAW images.
// ...
// 7 - User manually selects rectangles + presses CTRL+M on the 1st image from the 2nd output folder.
// 8 - User clicks OK to continue macro execution.
// 9 - The macro automatically selects rectangles on every image, calculates a, b, c, d parameters, calibrates it, and saves to disk (into the 3rd output folder).

// **********************************************************

NUMBER_OF_SWATCHES = 6;
REFERENCE_VALUES_FOR_GRAY_PATCHES = newArray(230, 184, 138, 92, 46, 11);

// Allow a user to disable/enable batch mode
Dialog.create("Antonello");
Dialog.addCheckbox("Batch Mode (faster processing)", true);
Dialog.show();
ENABLE_BATCH_MODE = Dialog.getCheckbox();


// **********************************************************

closeAllImages();

// 1 - Choose a folder with the registered images to process

while (true) {
	registeredDir = chooseRegisteredDir() ;
	
	if (countFilesInDir(registeredDir) == 0) {
		Dialog.create("Warning");
		Dialog.addMessage("The chosen folder doesn't contain any files!");
		Dialog.addMessage("Please, press OK and choose another folder.");
		Dialog.show();
		continue;
	}

	theNamesString = getFileNamesWithSpacesInDir(registeredDir);
	
	if (lengthOf(theNamesString) > 0) {
		Dialog.create("Warning");
		Dialog.addMessage("The next images contain spaces in their file names:");
		Dialog.addMessage(theNamesString);
		Dialog.addMessage("DCRAW plugin can't open such images.");
		Dialog.addMessage("Please, press OK and choose another folder.");
		Dialog.show();
	} else {
		break;
	}
}

// 2 - Choose a folder for the calibrated images
calibratedDir = getDirectory("Choose a folder for the calibrated images");


//...

// 5 - Calibrate images
calibrateImages(registeredDir, calibratedDir);

showMessage("Success", "Antonello-calib has finished its job");

// **********************************************************

function enableBatchMode(isEnable) {
	if (ENABLE_BATCH_MODE) {
		setBatchMode(isEnable);
	}
}

function calibrateImages(inputDir, outputDir) {
	imagesNames = getFileList(inputDir);
	imagesNames = Array.sort(imagesNames);

	firstImageIndex = 0;

	for (i = 0; i < imagesNames.length; i++) {
		imagePath = inputDir + imagesNames[i];
		if (!File.isDirectory(imagePath)) {
			firstImageIndex = i;
			break;
		}
	}

	// Open Results table with particular columns
	Table.create("Results");
	run("Set Measurements...", "mean bounding redirect=None decimal=3");
	
	// Open the 1st image and activate Rectangle tool
	firstImagePath = inputDir + imagesNames[firstImageIndex];
	open(firstImagePath);
	imageID = getImageID();
	setTool("rectangle");
	
	// Show a message to a user with the instructions
	waitForUser("Select 6 swatches: from WHITE to BLACK", "1. Select a small rectangle inside of a swatch\n2. Press CTRL+M\n3. Repeat previous steps for every swatch\n4. After finishing, click OK to CONTINUE");
	
	// Make sure there are proper number of rows in the Results table	
	while (Table.size("Results") != NUMBER_OF_SWATCHES) {
		waitForUser("Error: Wrong number of rows", "Please, make sure there are exactly 6 rows in the Results table\nClick OK to CONTINUE");
	}
	
	// Save the data from the Results table
	swatchesBX = Table.getColumn("BX");
	swatchesBY = Table.getColumn("BY");
	swatchesWidth = Table.getColumn("Width");
	swatchesHeight = Table.getColumn("Height");
	
	// Calculate the formula string for the first image
	swatchesMean = Table.getColumn("Mean");
	formulaString = fit3rdDegreePoly(swatchesMean);	

	// Calibrate image and save it to the 3rd output folder
	selectImage(imageID);
	imageTitle = getTitle();
	run("Select None");
	run("Macro...", formulaString);
	calibratedImagePath = outputDir + imageTitle;
	selectImage(imageID);
	saveAs("jpg", calibratedImagePath);
	selectImage(imageID);
	close();

	enableBatchMode(true);
	// Loop through every image and calibrate it
	for (j = firstImageIndex + 1; j < imagesNames.length; j++) {

		// Skip folders
		imagePath = inputDir + imagesNames[j];
		if (File.isDirectory(imagePath)) {
			continue;
		}

		Table.reset("Results");
		open(imagePath);
		imageID = getImageID();
	
		// Make rectangle selections and take measurements		
		for (k = 0; k < NUMBER_OF_SWATCHES; k++) {
			makeRectangle(swatchesBX[k], swatchesBY[k], swatchesWidth[k], swatchesHeight[k]);
			run("Measure");
		}

		// Calculate the formula string
		newMeanValues = Table.getColumn("Mean");
		formulaString = fit3rdDegreePoly(newMeanValues);

		// Calibrate image and save it to the 3rd output folder
		selectImage(imageID);
		imageTitle = getTitle();
		run("Select None");
		run("Macro...", formulaString);
		calibratedImagePath = outputDir + imageTitle;
		selectImage(imageID);
		saveAs("jpg", calibratedImagePath);
		selectImage(imageID);
		close();
	}
	
	enableBatchMode(false);

	close("Results");
}

function getFileNamesWithSpacesInDir(inputDir) {
	imagesNames = getFileList(inputDir);
	theString = "";

	for (i = 0; i < imagesNames.length; i++) {
	
		imagePath = inputDir + imagesNames[i];
	
		if (File.isDirectory(imagePath)) {
			continue;
		}

		isStringContainsSpace = (indexOf(imagesNames[i], " ") != -1);
		if (isStringContainsSpace) {
			theString += "\n" + imagesNames[i];
		}
	}
	
	return theString;
}

function countFilesInDir(inputDir) {
	imagesNames = getFileList(inputDir);
	filesCount = 0;

	for (i = 0; i < imagesNames.length; i++) {
	
		imagePath = inputDir + imagesNames[i];
	
		if (File.isDirectory(imagePath)) {
			continue;
		}
	
		filesCount++;
	}
	
	return filesCount;
}

function chooseRegisteredDir() {
	while (true) {
		registeredImagesDir = getDirectory("Choose a folder with the registered images");
		isSpaceCharacterDetected = (indexOf(registeredImagesDir, " ") != -1);

		if (isSpaceCharacterDetected) {
			
			Dialog.create("Warning");
			Dialog.addMessage("The chosen folder's path contains spaces!");
			Dialog.addMessage("DCRAW plugin can't open images in this folder.");
			Dialog.addMessage("Please, press OK and choose another folder.");
			Dialog.show();
			
		} else {
			break;
		}
	}

	return registeredImagesDir;
}

function resaveFirstFile(inputDir, outputDir) {
	imagesNames = getFileList(inputDir);

	for (i = 0; i < imagesNames.length; i++) {
	
		imagePath = inputDir + imagesNames[i];
	
		if (!File.isDirectory(imagePath)) {
			newImagePath = outputDir + imagesNames[i];
			open(imagePath);
			saveAs("tif", newImagePath);
			close();
			break;
		}
	}
}

function closeAllImages() {
	while (nImages > 0) {
		selectImage(nImages);
		close();
	}
}

function fit3rdDegreePoly(x) {
	Fit.doFit("3rd Degree Polynomial", x, REFERENCE_VALUES_FOR_GRAY_PATCHES);
	a = Fit.p(0);
	b = Fit.p(1);
	c = Fit.p(2);
	d = Fit.p(3);
	formulaString = "code=v=("+a+")+("+b+")*v+("+c+")*v*v+("+d+")*v*v*v";

	return formulaString;	
}
